<?php
/**
 * Helper Functions
 * SSCI Office Platform
 */

/**
 * Get translated string
 */
function trans(string $key, array $replace = []): string
{
    if (!isset($GLOBALS['language'])) {
        return $key;
    }
    return $GLOBALS['language']::get($key, $replace);
}

/**
 * Alias for trans()
 */
function __($key, array $replace = []): string
{
    return trans($key, $replace);
}

/**
 * Get config value
 */
function config(string $key, mixed $default = null): mixed
{
    if (!isset($GLOBALS['config'])) {
        return $default;
    }
    return $GLOBALS['config_obj']::get($key, $default);
}

/**
 * Get config file path
 */
function config_path(string $filename): string
{
    return SSCI_CONFIG . '/' . $filename;
}

/**
 * Get current language
 */
function current_language(): string
{
    if (!isset($GLOBALS['language'])) {
        return 'en';
    }
    return $GLOBALS['language']::getCurrentLanguage();
}

/**
 * Get text direction
 */
function text_direction(): string
{
    if (!isset($GLOBALS['language'])) {
        return 'ltr';
    }
    return $GLOBALS['language']::getDirection();
}

/**
 * Check if user is authenticated
 */
function is_authenticated(): bool
{
    if (!isset($GLOBALS['auth'])) {
        return false;
    }
    return $GLOBALS['auth']->isAuthenticated();
}

/**
 * Get current user
 */
function current_user(): ?array
{
    if (!isset($GLOBALS['auth']) || !is_authenticated()) {
        return null;
    }
    return $GLOBALS['auth']->getCurrentUser();
}

/**
 * Check if user has permission
 */
function has_permission(string $permission): bool
{
    if (!isset($GLOBALS['auth'])) {
        return false;
    }
    return $GLOBALS['auth']->hasPermission($permission);
}

/**
 * Check if user can access page
 */
function can_access_page(string $page): bool
{
    if (!isset($GLOBALS['auth'])) {
        return false;
    }
    return $GLOBALS['auth']->canAccessPage($page);
}

/**
 * Get notification
 */
function notify(string $title, string $message, string $type = 'info'): void
{
    if (!isset($GLOBALS['notification'])) {
        return;
    }
    $GLOBALS['notification']->createInAppNotification(
        current_user()['id'] ?? 0,
        $title,
        $message
    );
}

/**
 * Escape HTML output
 */
function escape(string $string): string
{
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Alias for escape()
 */
function e(string $string): string
{
    return escape($string);
}

/**
 * Load language
 */
function load_language(string $lang): void
{
    if (isset($GLOBALS['language'])) {
        $GLOBALS['language']::loadLanguage($lang);
    }
}

/**
 * Convert number to words (for amounts in contracts)
 */
function number_to_words(float $number): string
{
    $ones = ['', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine'];
    $teens = ['ten', 'eleven', 'twelve', 'thirteen', 'fourteen', 'fifteen', 'sixteen', 'seventeen', 'eighteen', 'nineteen'];
    $tens = ['', '', 'twenty', 'thirty', 'forty', 'fifty', 'sixty', 'seventy', 'eighty', 'ninety'];
    $scales = ['', 'thousand', 'million', 'billion', 'trillion'];

    if ($number == 0) {
        return 'zero';
    }

    $number = (int) $number;
    $groups = [];
    $groupIndex = 0;

    while ($number > 0) {
        $groups[] = $number % 1000;
        $number = (int) ($number / 1000);
        $groupIndex++;
    }

    $words = [];
    $groupIndex = count($groups) - 1;

    foreach (array_reverse($groups) as $group) {
        if ($group == 0) {
            $groupIndex--;
            continue;
        }

        $groupWords = [];
        $hundreds = (int) ($group / 100);
        $remainder = $group % 100;

        if ($hundreds > 0) {
            $groupWords[] = $ones[$hundreds] . ' hundred';
        }

        if ($remainder >= 20) {
            $tenDigit = (int) ($remainder / 10);
            $oneDigit = $remainder % 10;
            $groupWords[] = $tens[$tenDigit] . ($oneDigit > 0 ? ' ' . $ones[$oneDigit] : '');
        } elseif ($remainder >= 10) {
            $groupWords[] = $teens[$remainder - 10];
        } elseif ($remainder > 0) {
            $groupWords[] = $ones[$remainder];
        }

        if ($groupIndex > 0 && !empty($groupWords)) {
            $groupWords[] = $scales[$groupIndex];
        }

        $words = array_merge($words, $groupWords);
        $groupIndex--;
    }

    return implode(' ', $words);
}
